<?php

namespace Noptin\WooCommerce\Triggers;

/**
 * Card before expiry trigger.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Card before expiry trigger.
 *
 * @since 1.0.0
 */
class Saved_Card_Before_Expiry extends \Noptin_WooCommerce_Trigger {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function __construct() {
		add_action( 'noptin_woocommerce_daily_action', array( $this, 'maybe_trigger' ) );
		add_action( 'handle_noptin_woocommerce_saved_card_before_expiry_trigger', array( $this, 'trigger_for_token' ), 10, 2 );
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'woocommerce_saved_card_before_expiry';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'WooCommerce > Before Saved Card Expires', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'X days before a saved card expires', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_rule_description( $rule ) {
		$settings = $rule->trigger_settings;

		/* translators: %s Expiry days */
		return $this->get_days( $settings, __( 'Days before expiry: %s', 'noptin-woocommerce' ) );
	}

	/**
	 * Returns the days before expiry.
	 *
	 * @param array $settings The trigger settings.
	 * @param string $context The context.
	 * @return string
	 */
	public function get_days( $settings, $context ) {

		// Abort if days are not specified.
		if ( empty( $settings['days'] ) ) {
			return $this->get_description();
		}

		// Prepare days array, e.g, 1 day, 3 days, 10 days.
		$_days = array();

		foreach ( array_unique( wp_parse_id_list( $settings['days'] ) ) as $days ) {
			$_days[] = sprintf(
				/* translators: %d number of days */
				_n( '%d day', '%d days', $days, 'noptin-woocommerce' ),
				$days
			);
		}

		// Combine the days.
		$last_day = array_pop( $_days );

		if ( $_days ) {
			$days = sprintf(
				// translators: %s number of days, e.g, 1 day, 3 days, 10 days.
				__( '%1$s and %2$s', 'noptin-woocommerce' ),
				implode( ', ', $_days ),
				$last_day
			);
		} else {
			$days = $last_day;
		}

		// Generate the description.
		return sprintf(
			'%s <p class="description">%s</p>',
			$this->get_description(),
			sprintf( $context, '<code>' . $days . '</code>' )
		);
	}

	/**
     * Returns an array of known smart tags.
     *
     * @since 1.0.0
     * @return array
     */
    public function get_known_smart_tags() {

		return array_merge(
			parent::get_known_smart_tags(),
			$this->get_token_smart_tags(),
			$this->get_customer_smart_tags()
		);
    }

	/**
	 * Returns token smart tags.
	 *
	 * @return array
	 */
	public function get_token_smart_tags() {

		return array(

			'card.id'                 => array(
				'description'       => __( 'Card ID', 'noptin-woocommerce' ),
				'example'           => 'card.id',
				'conditional_logic' => 'number',
			),

			'card.display_name'       => array(
				'description'       => __( "Card's display name", 'noptin-woocommerce' ),
				'example'           => 'card.display_name',
				'conditional_logic' => 'string',
			),

			'card.gateway_id'         => array(
				'description'       => __( 'Gateway ID', 'noptin-woocommerce' ),
				'example'           => 'card.gateway_id',
				'conditional_logic' => 'string',
			),

			'subscription.is_default' => array(
				'description'       => __( 'Is default card', 'noptin-woocommerce' ),
				'example'           => 'card.gateway_id',
				'conditional_logic' => 'string',
				'options'           => array(
					'yes' => __( 'Yes', 'noptin-woocommerce' ),
					'no'  => __( 'No', 'noptin-woocommerce' ),
				),
			),

			'card.type'               => array(
				'description'       => __( 'Card type', 'noptin-woocommerce' ),
				'example'           => 'card.type',
				'conditional_logic' => 'string',
			),

			'card.expiry_year'        => array(
				'description'       => __( "Card's expiry year", 'noptin-woocommerce' ),
				'example'           => 'card.expiry_year',
				'conditional_logic' => 'number',
			),

			'card.expiry_month'       => array(
				'description'       => __( "Card's expiry month", 'noptin-woocommerce' ),
				'example'           => 'card.expiry_month',
				'conditional_logic' => 'number',
			),

			'card.expire_days'        => array(
				'description'       => __( 'Number of days before the card expires', 'noptin-woocommerce' ),
				'example'           => 'card.expiry_month',
				'conditional_logic' => 'number',
			),

			'card.last_four'          => array(
				'description'       => __( "Card's last four digits", 'noptin-woocommerce' ),
				'example'           => 'card.last_four',
				'conditional_logic' => 'number',
			),
		);
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array_merge(
			array(
				'days' => array(
					'el'          => 'input',
					'label'       => __( 'Days before expiry', 'noptin-woocommerce' ),
					'description' => __( 'Enter a comma-separated list of days before expiry for which this trigger will fire.', 'noptin-woocommerce' ),
					'placeholder' => '1, 3, 7',
				),
			),
			parent::get_settings()
		);

	}

	/**
	 * Fetch cards that are about to expire.
	 *
	 * @param int[] $days The days before expiry.
	 * @return int[] $token_ids The token ids.
	 */
	public function get_cards( $days ) {
		global $wpdb;

		$years = array();

		foreach ( $days as $day ) {

			// Add the days to todays date and save it if it is the last day of the resulting month.
			$date = gmdate( 'Y-m-d', strtotime( "+{$day} days" ) );

			if ( gmdate( 't', strtotime( $date ) ) === gmdate( 'j', strtotime( $date ) ) ) {
				$year = gmdate( 'Y', strtotime( $date ) );

				if ( ! isset( $years[ $year ] ) ) {
					$years[ $year ] = array();
				}

				$years[ $year ][] = gmdate( 'm', strtotime( $date ) );
			}
		}

		$token_ids = array();

		foreach ( $years as $year => $months ) {

			$year_tokens = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT token_id FROM {$wpdb->prefix}woocommerce_payment_tokens as tokens
					LEFT JOIN {$wpdb->payment_tokenmeta} AS m1 ON tokens.token_id = m1.payment_token_id
					LEFT JOIN {$wpdb->payment_tokenmeta} AS m2 ON tokens.token_id = m2.payment_token_id
					WHERE type = 'CC'
					AND m1.meta_key = 'expiry_year'
					AND m1.meta_value = %s
					AND m2.meta_key = 'expiry_month'
					AND m2.meta_value IN (" . implode( ',', array_fill( 0, count( $months ), '%s' ) ) . ')',
					$year,
					...$months
				)
			);

			$token_ids = array_merge( $token_ids, wp_parse_id_list( $year_tokens ) );
		}

		return array_unique( $token_ids );

	}

	/**
	 * Maybe triggers an action.
	 */
	public function maybe_trigger() {

		// Loop through all rules.
		foreach ( $this->get_rules() as $rule ) {

            // Prepare the rule.
            $rule = noptin()->automation_rules->prepare_rule( $rule );

			if ( ! isset( $rule->trigger_settings['days'] ) ) {
				continue;
			}

			// Fetch cards.
			$cards = $this->get_cards( wp_parse_id_list( $rule->trigger_settings['days'] ) );

			// Abort if no cards are found.
			if ( empty( $cards ) ) {
				continue;
			}

			// Trigger for each card.
			foreach ( $cards as $index => $token_id ) {

				if ( $index > 10 ) {
					$schedule_for = floor( $index / 10 ) * MINUTE_IN_SECONDS;
					schedule_noptin_background_action( time() + $schedule_for, 'handle_noptin_woocommerce_saved_card_before_expiry_trigger', $token_id, $rule->id );
				} else {
					schedule_noptin_background_action( time() + 30, 'handle_noptin_woocommerce_saved_card_before_expiry_trigger', $token_id, $rule->id );
				}
			}
        }

	}

	/**
	 * Triggers a token event.
	 *
	 * @param int $token_id The token id.
	 * @param int $rule_id The rule id.
	 */
	public function trigger_for_token( $token_id, $rule_id ) {

		/** @var \WC_Payment_Token_CC $token */
		$token = \WC_Payment_Tokens::get( $token_id );

		if ( ! $token || ! $token->get_id() ) {
			return;
		}

		$customer = new \WC_Customer( $token->get_user_id() );

		if ( ! $customer->get_id() ) {
			return;
		}

		// Prepare the trigger.
		$args         = $this->before_trigger_wc( false, $customer );
		$expire_year  = $token->get_expiry_year();
		$expire_month = $token->get_expiry_month();
		$last_day     = gmdate( 't', strtotime( "{$expire_year}-{$expire_month}-01" ) );
		$expire_date  = strtotime( "{$expire_year}-{$expire_month}-{$last_day} 23:59:59" ); // Get timestamp of last day of the month

		$args = array_replace(
			$args,
			array(
				'email'             => $customer->get_email(),
				'card.id'           => $token->get_id(),
				'card.display_name' => $token->get_display_name(),
				'card.gateway_id'   => $token->get_gateway_id(),
				'card.is_default'   => $token->get_is_default() ? 'yes' : 'no',
				'card.type'         => $token->get_card_type(),
				'card.expiry_year'  => $token->get_expiry_year(),
				'card.expiry_month' => $token->get_expiry_month(),
				'card.expire_days'  => ceil( ( $expire_date - time() ) / DAY_IN_SECONDS ),
				'card.expire_date'  => gmdate( 'Y-m-d', $expire_date ),
				'card.last_four'    => $token->get_last4(),
				'rule_id'           => $rule_id,
			)
		);

		$this->trigger( $customer, $args );

		$this->after_trigger_wc( $args );
	}

	/**
	 * Triggers action callbacks.
	 *
	 * @since 1.2.8
	 * @param \WC_Customer $subject The subject.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function trigger( $subject, $args ) {

		if ( empty( $args['rule_id'] ) ) {
			return;
		}

		$args = $this->prepare_trigger_args( $subject, $args );
		$rule = new \Noptin_Automation_Rule( $args['rule_id'] );

		// Retrieve the action.
		$action = noptin()->automation_rules->get_action( $rule->action_id );
		if ( empty( $action ) ) {
			return;
		}

		// Set the current email.
		$GLOBALS['current_noptin_email'] = $this->get_subject_email( $subject, $rule, $args );

		// Are we delaying the action?
		$delay = $rule->get_delay();

		if ( $delay > 0 ) {
			do_action( 'noptin_delay_automation_rule_execution', $rule, $args, $delay );
			return;
		}

		// Ensure that the rule is valid for the provided args.
		if ( $this->is_rule_valid_for_args( $rule, $args, $subject, $action ) ) {
			$action->maybe_run( $subject, $rule, $args );
		}

	}

	/**
	 * Serializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return false|array
	 */
	public function serialize_trigger_args( $args ) {
		return array(
			'card.id' => $args['card.id'],
			'rule_id' => $args['rule_id'],
		);
	}

	/**
	 * Unserializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return array|false
	 */
	public function unserialize_trigger_args( $args ) {

		/** @var \WC_Payment_Token_CC $token */
		$token = \WC_Payment_Tokens::get( $args['card.id'] );

		if ( ! $token ) {
			throw new \Exception( 'The card no longer exists' );
		}

		$customer = new \WC_Customer( $token->get_user_id() );

		if ( ! $customer->get_id() ) {
			throw new \Exception( 'The customer no longer exists' );
		}

		// Prepare the trigger.
		$args         = $this->before_trigger_wc( false, $customer );
		$expire_year  = $token->get_expiry_year();
		$expire_month = $token->get_expiry_month();
		$last_day     = gmdate( 't', strtotime( "{$expire_year}-{$expire_month}-01" ) );
		$expire_date  = strtotime( "{$expire_year}-{$expire_month}-{$last_day} 23:59:59" ); // Get timestamp of last day of the month

		$args = array_replace(
			$args,
			array(
				'email'             => $customer->get_email(),
				'card.id'           => $token->get_id(),
				'card.display_name' => $token->get_display_name(),
				'card.gateway_id'   => $token->get_gateway_id(),
				'card.is_default'   => $token->get_is_default() ? 'yes' : 'no',
				'card.type'         => $token->get_card_type(),
				'card.expiry_year'  => $token->get_expiry_year(),
				'card.expiry_month' => $token->get_expiry_month(),
				'card.expire_days'  => ceil( ( $expire_date - time() ) / DAY_IN_SECONDS ),
				'card.expire_date'  => gmdate( 'Y-m-d', $expire_date ),
				'card.last_four'    => $token->get_last4(),
				'rule_id'           => $args['rule_id'],
			)
		);

		// Prepare the trigger args.
		return $this->prepare_trigger_args( $customer, $args );
	}
}
