<?php

namespace Noptin\Addons_Pack\Users\Actions;

use Noptin\Addons_Pack\Users\Custom_Fields;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Creates a new user.
 *
 * @since 1.10.0
 */
class Add_User extends Abstract_Action {

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'add_user';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'User > Create/Update User', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'Create or pdate user account', 'noptin-addons-pack' );
	}

	/**
	 * Retrieve the actions's rule table description.
	 *
	 * @since 1.11.9
	 * @param Noptin_Automation_Rule $rule
	 * @return array
	 */
	public function get_rule_table_description( $rule ) {
		$settings = $rule->action_settings;

		// Ensure we have a role.
		if ( empty( $settings['role'] ) ) {
			return sprintf(
				'<span class="noptin-rule-error">%s</span>',
				esc_html__( 'Error: Role not specified', 'noptin-addons-pack' )
			);
		}

		$roles = array_map( 'translate_user_role', wp_list_pluck( array_reverse( get_editable_roles() ), 'name' ) );
		$role  = isset( $roles[ $settings['role'] ] ) ? $roles[ $settings['role'] ] : $settings['role'];
		$meta  = array(
			__( 'Role', 'noptin-addons-pack' ) => $role,
		);

		return $this->rule_action_meta( $meta, $rule );
	}

	/**
	 * @return array
	 */
	public function get_user_fields() {

		$fields = array(
			'first_name'  => __( 'First Name', 'noptin-addons-pack' ),
			'last_name'   => __( 'Last Name', 'noptin-addons-pack' ),
			'description' => __( 'Bio', 'noptin-addons-pack' ),
			'role'        => array(
				'label'       => __( 'Role', 'noptin-addons-pack' ),
				'options'     => array_merge(
					array( '' => __( 'Select a role', 'noptin-addons-pack' ) ),
					array_map( 'translate_user_role', wp_list_pluck( array_reverse( get_editable_roles() ), 'name' ) )
				),
				'description' => __( 'The new user\'s role. Ignored when updating an existing account.', 'noptin-addons-pack' ),
			),
		);

		foreach ( Custom_Fields::get() as $field_name => $field ) {
			$fields[ 'user_cf_' . $field_name ] = $field['label'];

			if ( ! empty( $field['options'] ) ) {
				$fields[ 'user_cf_' . $field_name ] = array(
					'label'   => $field['label'],
					'options' => $field['options'],
				);
			}
		}

		return $fields;
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		$settings = array();

		foreach ( $this->get_user_fields() as $key => $field ) {

			$settings[ $key ] = array(
				'el'          => 'input',
				'label'       => is_array( $field ) ? $field['label'] : $field,
				'placeholder' => __( 'Enter a value or leave blank to ignore', 'noptin-addons-pack' ),
				'default'     => '',
				'description' => sprintf(
					/* translators: %1: Opening link, %2 closing link tag. */
					esc_html__( 'You can use %1$s smart tags %2$s to assign dynamic values.', 'noptin-addons-pack' ),
					'<a href="#TB_inline?width=0&height=550&inlineId=noptin-automation-rule-smart-tags" class="thickbox">',
					'</a>'
				),
			);

			if ( is_array( $field ) && ! empty( $field['options'] ) ) {
				$settings[ $key ]['el']      = 'select';
				$settings[ $key ]['options'] = $field['options'];
				unset( $settings[ $key ]['placeholder'] );
				unset( $settings[ $key ]['description'] );

				if ( isset( $field['description'] ) ) {
					$settings[ $key ]['description'] = $field['description'];
				}
			}
		}

		return $settings;
	}

	/**
	 * Returns whether or not the action can run (dependancies are installed).
	 *
	 * @since 1.3.3
	 * @param mixed $subject The subject.
	 * @param Noptin_Automation_Rule $rule The automation rule used to trigger the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return bool
	 */
	public function can_run( $subject, $rule, $args ) {
		return ! ! $this->get_subject_email( $subject, $rule, $args );
	}

	/**
	 * Updates the user's role.
	 *
	 * @since 1.10.0
	 * @param mixed $subject The subject.
	 * @param \Noptin_Automation_Rule $rule The automation rule used to trigger the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function run( $subject, $rule, $args ) {

		$user_info = array( 'meta_input' => array() );
		$settings  = $rule->action_settings;

		/** @var \Noptin_Automation_Rules_Smart_Tags $smart_tags */
		$smart_tags = $args['smart_tags'];

		foreach ( array_keys( $this->get_user_fields() ) as $key ) {

			// Abort if not set.
			if ( empty( $settings[ $key ] ) ) {
				continue;
			}

			$value = $smart_tags->replace_in_html( $settings[ $key ] );

			// Abort if empty.
			if ( '' === $value ) {
				continue;
			}

			// Handle custom fields.
			if ( 0 === strpos( $key, 'user_cf_' ) ) {
				$user_info['meta_input'][ substr( $key, 8 ) ] = $value;
				continue;
			}

			// Normal fields.
			$user_info[ $key ] = $value;
		}

		if ( empty( $user_info['meta_input'] ) ) {
			unset( $user_info['meta_input'] );
		}

		// Insert or update the user.
		$user = $this->get_user( $subject, $rule, $args );

		if ( ! empty( $user ) && ! is_wp_error( $user ) ) {

			// Unset user role.
			unset( $user_info['role'] );

			// Update the user.
			$user_info['ID'] = $user->ID;
			return wp_update_user( $user_info );
		}

		$user_info['user_email'] = $this->get_subject_email( $subject, $rule, $args );
		$user_info['user_login'] = sanitize_user( $this->generate_username( $user_info['user_email'], $user_info ) );
		$user_info['user_pass']  = wp_generate_password();

		// Validate username.
		if ( empty( $user_info['user_login'] ) || ! validate_username( $user_info['user_login'] ) ) {
			return;
		}

		// Create the user.
		wp_insert_user( $user_info );
	}

	/**
	 * Create a unique username for a new user.
	 *
	 * @since 1.0.0
	 * @param string $email New user email address.
	 * @param array  $new_user_args Array of new user args, maybe including first and last names.
	 * @param string $suffix Append string to username to make it unique.
	 * @return string Generated username.
	 */
	private function generate_username( $email, $new_user_args = array(), $suffix = '' ) {
		$username_parts = array();

		if ( isset( $new_user_args['first_name'] ) ) {
			$username_parts[] = sanitize_user( $new_user_args['first_name'], true );
		}

		if ( isset( $new_user_args['last_name'] ) ) {
			$username_parts[] = sanitize_user( $new_user_args['last_name'], true );
		}

		// Remove empty parts.
		$username_parts = array_filter( $username_parts );

		// If there are no parts, e.g. name had unicode chars, or was not provided, fallback to email.
		if ( empty( $username_parts ) ) {
			$email_parts    = explode( '@', $email );
			$email_username = $email_parts[0];

			// Exclude common prefixes.
			if ( in_array(
				$email_username,
				array(
					'sales',
					'hello',
					'mail',
					'contact',
					'info',
				),
				true
			) ) {
				// Get the domain part.
				$email_username = $email_parts[1];
			}

			$username_parts[] = sanitize_user( $email_username, true );
		}

		$username = strtolower( implode( '.', $username_parts ) );

		if ( $suffix ) {
			$username .= $suffix;
		}

		/**
		 * WordPress 4.4 - filters the list of blocked usernames.
		 *
		 * @since 3.7.0
		 * @param array $usernames Array of blocked usernames.
		 */
		$illegal_logins = (array) apply_filters( 'illegal_user_logins', array() );

		// Stop illegal logins and generate a new random username.
		if ( in_array( strtolower( $username ), array_map( 'strtolower', $illegal_logins ), true ) ) {
			$new_args = array();

			$new_args['first_name'] = apply_filters(
				'noptin_generated_user_username',
				'noptin_user_' . zeroise( wp_rand( 0, 9999 ), 4 ),
				$email,
				$new_user_args,
				$suffix
			);

			return $this->generate_username( $email, $new_args, $suffix );
		}

		if ( username_exists( $username ) ) {
			// Generate something unique to append to the username in case of a conflict with another user.
			$suffix = '-' . zeroise( wp_rand( 0, 9999 ), 4 );
			return $this->generate_username( $email, $new_user_args, $suffix );
		}

		return apply_filters( 'noptin_generated_user_username', $username, $email, $new_user_args, $suffix );
	}
}
