<?php

namespace Noptin\Addons_Pack\Emails;

/**
 * Emails API: Periodic email.
 *
 * Sends the same email to all subscribers on a regular basis.
 *
 * @since   1.10.2
 * @package Noptin
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Sends the same email to all subscribers on a regular basis.
 *
 * @since 1.10.2
 * @internal
 * @ignore
 */
class Periodic_Email extends \Noptin_Automated_Email_Type {

	/**
	 * @var string
	 */
	public $category = 'Mass Mail';

	/**
	 * @var string
	 */
	public $type = 'periodic';

	/**
	 * @var string
	 */
	public $notification_hook = 'noptin_send_periodic_email';

	/**
	 * Registers relevant hooks.
	 *
	 */
	public function add_hooks() {
		parent::add_hooks();
		add_filter( 'noptin_automated_email_supports_timing', array( $this, 'supports_timing' ), 10, 2 );
		add_filter( 'noptin_automation_is_mass_mail', array( $this, 'is_mass_mail' ), 10, 2 );
	}

	/**
	 * Returns true if the email supports timing.
	 */
	public function supports_timing( $supports, $email_type ) {

		if ( $email_type === $this->type ) {
			return false;
		}

		return $supports;
	}

	/**
	 * Returns true if the email is a mass mail.
	 */
	public function is_mass_mail( $is_mass_mail, $email_type ) {

		if ( $email_type === $this->type ) {
			return true;
		}

		return $is_mass_mail;
	}

	/**
	 * Retrieves the automated email type name.
	 *
	 */
	public function get_name() {
		return __( 'Periodic', 'noptin-addons-pack' );
	}

	/**
	 * Retrieves the automated email type description.
	 *
	 */
	public function get_description() {
		return __( 'Automatically send your subscribers, users, or customers an email every X days.', 'noptin-addons-pack' );
	}

	/**
	 * Retrieves the automated email type image.
	 *
	 * @deprecated 1.11.0
	 */
	public function the_image() {
		echo '<svg xmlns="http://www.w3.org/2000/svg" fill="#3f9ef4" viewBox="0 0 122.88 122.88"><path d="M61.44,0A61.46,61.46,0,1,1,18,18,61.21,61.21,0,0,1,61.44,0ZM32.22,79.39,52.1,59.46,32.22,43.25V79.39ZM54.29,61.24,33.79,81.79H88.91L69.33,61.24l-6.46,5.51h0a1.42,1.42,0,0,1-1.8,0l-6.78-5.53Zm17.18-1.82L90.66,79.55V43.07L71.47,59.42ZM34,41.09l27.9,22.76L88.65,41.09Zm65.4-17.64a53.72,53.72,0,1,0,15.74,38,53.56,53.56,0,0,0-15.74-38Z"/></svg>';
	}

	/**
	 * Returns the image URL or dashicon for the automated email type.
	 *
	 * @return string|array
	 */
	public function get_image() {
		return array(
			'icon' => 'calendar',
			'fill' => '#3f9ef4',
		);
	}

	/**
	 * Returns the next send date.
	 *
	 */
	public function default_next_send() {
		return gmdate( 'Y-m-d H:i:s', strtotime( 'tomorrow 07:00' ) );
	}

	/**
	 * Returns the default x days.
	 */
	public function default_x_days() {
		return '30';
	}

	/**
	 * Returns the default time.
	 *
	 */
	public function default_time() {
		return '07:00';
	}

	/**
	 * Displays a metabox.
	 *
	 * @param \Noptin_Automated_Email $campaign
	 */
	public function render_metabox( $campaign ) {

		$next_send = (string) $campaign->get( 'next_send' );
		$x_days    = (string) $campaign->get( 'x_days' );
		$time      = (string) $campaign->get( 'time' );

		?>

		<!-- Send this email every ___ days at ___ starting ___ -->
		<p>
			<label>
				<?php esc_html_e( 'Send this email every', 'noptin-addons-pack' ); ?>
				<input type="number" name="noptin_email[x_days]" value="<?php echo esc_attr( $x_days ); ?>" min="1" max="365" step="1" style="margin-bottom: 10px; width: 50px;" required>
			</label>
			<label>
				<?php esc_html_e( 'days at', 'noptin-addons-pack' ); ?>
				<input type="time" name="noptin_email[time]" value="<?php echo esc_attr( $time ); ?>" style="margin-bottom: 10px;" required>
			</label>
		</p>

		<p>
			<label>
				<strong class="noptin-label-span"><?php esc_html_e( 'Next send is', 'noptin-addons-pack' ); ?></strong>

				<span class="noptin-post-digest-frequency noptin-inline-block" style="margin-bottom: 10px;">
					<input type="text" name="noptin_email[next_send]" value="<?php echo esc_attr( $next_send ); ?>" class="noptin-inline-block" style="margin-bottom: 10px;" required>
				</span>

			</label>
		</p>

		<?php

	}

	/**
	 * Fires after an automation is saved.
	 *
	 * @param \Noptin_Automated_Email $campaign
	 */
	public function on_save_campaign( $campaign ) {

		// If next send is empty, set it to the default.
		$next_send = $campaign->get( 'next_send' );
		if ( empty( $next_send ) ) {
			$days = $campaign->get( 'x_days' );

			if ( empty( $days ) ) {
				$days                        = $this->default_x_days();
				$campaign->options['x_days'] = $days;
			}

			$time = $campaign->get( 'time' );

			if ( empty( $time ) ) {
				$time                      = $this->default_time();
				$campaign->options['time'] = $time;
			}

			// Add days to today.
			$next_send                      = gmdate( 'Y-m-d H:i:s', strtotime( "+$days days $time" ) );
			$campaign->options['next_send'] = $next_send;
			return $campaign->save();
		}

		$this->schedule_campaign( $campaign );
	}

	/**
	 * Schedules the next send for a given campain.
	 *
	 * @param \Noptin_Automated_Email $campaign
	 * @param int $is_saving
	 */
	public function schedule_campaign( $campaign ) {

		// Clear scheduled task.
		\Noptin\Addons_Pack\Tasks\Main::delete_scheduled_task( $this->notification_hook, array( $campaign->id ) );

		// Abort if the campaign is not active.
		if ( ! $campaign->can_send() ) {
			return;
		}

		// Schedule the next send.
		$next_send = $campaign->get( 'next_send' );

		if ( empty( $next_send ) ) {
			return;
		}

		// Calculate seconds.
		$next_send = strtotime( $next_send ) - time() - ( (float) get_option( 'gmt_offset' ) * HOUR_IN_SECONDS );

		// Schedule the next send.
		\Noptin\Addons_Pack\Tasks\Main::schedule_task( $this->notification_hook, array( $campaign->id ), $next_send );

	}

	/**
	 * (Maybe) Send out post digests.
	 *
	 * @param int $campaign_id
	 * @param string $key
	 */
	public function maybe_send_notification( $campaign_id ) {

		// Get the campaign.
		$campaign = new \Noptin_Automated_Email( $campaign_id );

		// Ensure that the campaign is still published.
		if ( ! $campaign->can_send() ) {
			return;
		}

		// Reschedule next send.
		$campaign->options['next_send'] = '';
		$this->on_save_campaign( $campaign );

		$should_send_hook = $this->notification_hook . '_should_send';

		// Remove all filters.
		remove_all_filters( $should_send_hook );

		do_action( 'noptin_before_send_periodic_email', $campaign );

		$type    = $campaign->get_email_type();
		$content = $campaign->get_content( $type );

		// Parse paragraphs.
		if ( 'normal' === $type ) {
			$content = wpautop( trim( $content ) );
		}

		$this->before_send( $campaign );

		// Prepare campaign args.
		$args = array_merge(
			$campaign->options,
			array(
				'parent_id'         => $campaign->id,
				'status'            => 'publish',
				'subject'           => noptin_parse_email_subject_tags( $campaign->get_subject(), true ),
				'heading'           => noptin_parse_email_content_tags( $campaign->get( 'heading' ), true ),
				'content_' . $type  => trim( noptin_parse_email_content_tags( $content, true ) ),
				'subscribers_query' => array(),
				'preview_text'      => noptin_parse_email_content_tags( $campaign->get( 'preview_text' ), true ),
				'footer_text'       => noptin_parse_email_content_tags( $campaign->get( 'footer_text' ), true ),
				'custom_title'      => sprintf( /* translators: %1 campaign name, %2 campaign date */ __( '%1$s [%2$s]', 'noptin-addons-pack' ), esc_html( $campaign->name ), date_i18n( get_option( 'date_format' ) ) ),
			)
		);

		// Check if we should send.
		if ( ! apply_filters( $should_send_hook, true, $args, $campaign ) ) {
			return;
		}

		// Remove unrelated content.
		foreach ( array( 'content_normal', 'content_plain_text', 'content_raw_html' ) as $content_type ) {
			if ( 'content_' . $type !== $content_type ) {
				unset( $args[ $content_type ] );
			}
		}

		// Prepare the newsletter.
		$newsletter = new \Noptin_Newsletter_Email( $args );
		$newsletter->save();

		// Clear environment.
		$this->after_send( $campaign );

	}

	/**
	 * Filters automation summary.
	 *
	 * @param string $about
	 * @param \Noptin_Automated_Email $campaign
	 */
	public function about_automation( $about, $campaign ) {

		// Prepare time.
		$time = $campaign->get( 'time' );

		if ( empty( $time ) ) {
			$time = '07:00';
		}

		// Convert time to a readable format.
		$time = date_i18n( get_option( 'time_format' ), strtotime( $time ) );

		// Days between sends.
		$x_days = (int) $campaign->get( 'x_days' );

		// Next send.
		$next_send = $campaign->get( 'next_send' );

		if ( $next_send ) {
			$next_send = strtotime( $next_send );
			$next_send = '<p class="noptin-list-table-misc noptin-tip" title="' . esc_attr( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $next_send ) ) . '">' . esc_html(
				sprintf(
					// Translators: human friendly diff time.
					__( 'Next send in %s', 'noptin-addons-pack' ),
					human_time_diff( $next_send - ( (float) get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) )
				)
			) . '</p>';
		}

		// Do not display the next send time if the campaign is paused.
		if ( ! $campaign->can_send() ) {
			$next_send = '';
		}

		return sprintf(
			// translators: %1 is the days, %2 is the time.
			_n( 'Sends every %1$s day at %2$s', 'Sends every %1$s days at %2$s', $x_days, 'noptin-addons-pack' ),
			$x_days,
			esc_html( $time )
		) . $next_send;
	}

	/**
	 * Sends a test email.
	 *
	 * @param Noptin_Automated_Email $campaign
	 * @param string $recipient
	 * @return bool Whether or not the test email was sent
	 */
	public function send_test( $campaign, $recipient ) {

		$this->prepare_test_data( $campaign );

		// Maybe set related subscriber.
		$subscriber = noptin_get_subscriber( sanitize_email( $recipient ) );

		if ( $subscriber->exists() ) {
			$this->subscriber = $subscriber;
		}

		return $this->send( $campaign, 'test', array( sanitize_email( $recipient ) => false ) );

	}

}
