<?php

namespace Noptin\Addons_Pack\Custom_Fields;

/**
 * Register custom fields based triggers and actions.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Registers custom fields based triggers and actions.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		add_filter( 'noptin_automation_rules_load', array( $this, 'load_triggers_and_actions' ) );
		add_filter( 'noptin_subscriber_sending_options', array( $this, 'add_sending_options' ), 10, 2 );
	}

	/**
	 * Loads automation rule triggers.
	 *
	 * @param \Noptin_Automation_Rules $rules
	 */
	public function load_triggers_and_actions( $rules ) {

		// Loop through all props.
		foreach ( get_noptin_subscriber_filters() as $merge_tag => $options ) {

			// Skip if no options.
			if ( empty( $options['options'] ) ) {
				continue;
			}

			$label = $options['label'];

			if ( 'confirmed' === $merge_tag ) {
				$label = __( 'Email confirmation status', 'noptin-addons-pack' );
			}

			// Add trigger.
			if ( ! empty( $options['is_multiple'] ) ) {
				$rules->add_trigger( new Triggers\Add_To_Field( $merge_tag, $label ) );
				$rules->add_trigger( new Triggers\Remove_From_Field( $merge_tag, $label ) );
				$rules->add_action( new Actions\Add_To_Field( $merge_tag, $label ) );
				$rules->add_action( new Actions\Remove_From_Field( $merge_tag, $label ) );
			} else {
				$rules->add_trigger( new Triggers\Change_Field( $merge_tag, $label ) );
				$rules->add_action( new Actions\Change_Field( $merge_tag, $label ) );
			}
		}

		// Tagging an untagging.
		$rules->add_trigger( new Triggers\Add_To_Field( 'tags', __( 'Tag', 'noptin-addons-pack' ) ) );
		$rules->add_trigger( new Triggers\Remove_From_Field( 'tags', __( 'Tag', 'noptin-addons-pack' ) ) );
		$rules->add_action( new Actions\Add_To_Field( 'tags', __( 'Tag', 'noptin-addons-pack' ) ) );
		$rules->add_action( new Actions\Remove_From_Field( 'tags', __( 'Tag', 'noptin-addons-pack' ) ) );
	}

	/**
	 * Adds sending options for custom fields.
	 *
	 * @param array $fields
	 */
	public function add_sending_options( $fields ) {

		foreach ( get_noptin_subscriber_filters() as $key => $filter ) {

			// Skip status since emails can only be sent to active subscribers.
			if ( 'status' === $key ) {
				continue;
			}

			$fields[ $key ] = array(
				'label'       => $filter['label'],
				'type'        => 'select',
				'options'     => array_replace(
					array(
						'' => __( 'Any', 'noptin-addons-pack' ),
					),
					$filter['options']
				),
				'description' => ( empty( $filter['description'] ) || $filter['label'] === $filter['description'] ) ? '' : $filter['description'],
			);

			if ( 2 < count( $filter['options'] ) || $filter['is_multiple'] ) {
				$fields[ $key ]['select2']     = 'true';
				$fields[ $key ]['options']     = $filter['options'];
				$fields[ $key ]['multiple']    = 'true';
				$fields[ $key ]['placeholder'] = __( 'Optional. Leave blank to send to all', 'noptin-addons-pack' );

				if ( $filter['is_multiple'] ) {
					$fields[ $key . '_not' ] = array_merge(
						$fields[ $key ],
						array(
							'label'       => sprintf(
								// translators: %s is the filter label.
								__( '%s - Exclude', 'noptin-addons-pack' ),
								$filter['label']
							),
							'description' => '',
						)
					);
				}
			}
		}

		$fields['tags'] = array(
			'label'       => __( 'Tags', 'noptin-addons-pack' ),
			'type'        => 'token',
			'description' => __( 'Optional. Filter recipients by their tags.', 'noptin-addons-pack' ),
		);

		$fields['tags_not'] = array(
			'label'       => __( 'Tags - Exclude', 'noptin-addons-pack' ),
			'type'        => 'token',
			'description' => __( 'Optional. Exclude recipients by their tags.', 'noptin-addons-pack' ),
		);

		return $fields;
	}
}
