<?php
/**
 * Noptin.com Custom Post Type Notifications Class.
 *
 * @package Noptin\noptin.com
 * @since   1.5.1
 */

defined( 'ABSPATH' ) || exit;

/**
 * Noptin_Custom_Post_Types Class
 *
 * @since 1.5.1
 * @ignore
 */
class Noptin_Custom_Post_Types {

	/**
	 * Init variables.
	 *
	 * @since       1.0.0
	 */
	public function __construct() {

		// Display metabox.
		add_action( 'noptin_automated_email_post_notifications_options', array( $this, 'render_metabox' ) );
		add_action( 'noptin_automated_email_post_digest_options', array( $this, 'render_metabox' ) );

		// Filter recipients.
		add_filter( 'noptin_get_email_prop_post_type', array( $this, 'filter_campaign_post_type' ), 10, 2 );
		add_filter( 'noptin_new_post_notification_valid_for_post', array( $this, 'filter_valid_for_post' ), 10, 3 );

		// Misc.
		add_filter( 'noptin_post_digest_merge_tag_query', array( $this, 'filter_post_digest_query' ), 10, 3 );
    }

	/**
     * Post types select box.
     *
     * @param  Noptin_Newsletter_Email $campaign
     * @since  1.0.0
     * @return string
     */
    public function render_metabox( $campaign ) {

		$post_type = $this->get_campaign_post_type( $campaign );
		$authors   = $campaign->get( 'noptin-ap-authors' );
		$language  = $campaign->get( 'noptin-ap-language' );

		$available_languages = noptin_get_available_languages();
		?>

		<p class="noptin-automation-post-type">

			<label style="display: block;">
				<strong><?php esc_html_e( 'Post Type', 'noptin-addons-pack' ); ?></strong>
				<select name="noptin_email[noptin-ap-post-type]" style="width: 100%;" class="noptin-ap-post-type-select">
					<option value="0" <?php selected( empty( $post_type ) ); ?> disabled><?php esc_html_e( 'Select a post type', 'noptin-addons-pack' ); ?></option>
					<?php foreach ( noptin_get_post_types() as $id => $name ) : ?>
						<option value="<?php echo esc_attr( $id ); ?>" <?php selected( $id, $post_type ); ?>><?php echo esc_html( $name ); ?></option>
					<?php endforeach; ?>
				</select>

				<?php if ( 'post_digest' === $campaign->type ) : ?>
					<span class="noptin-help-text">
						<?php esc_html_e( 'The [[post_digest]] merge tag will display this post type if you do not set a post type in the merge tag.', 'noptin-addons-pack' ); ?>
						<a href="https://noptin.com/guide/sending-emails/new-post-notifications/#set-digest-post-types" target="_blank"><?php esc_html_e( 'Learn more', 'noptin-addons-pack' ); ?></a>
					</span>
				<?php else : ?>
					<span class="noptin-help-text">
						<?php esc_html_e( 'New post notifications will only send for this post type.', 'noptin-addons-pack' ); ?>
					</span>
				<?php endif; ?>

			</label>

		</p>

		<?php if ( 'post_digest' !== $campaign->type ) : ?>
			<p class="noptin-automation-post-type">

				<label style="display: block;">
					<strong><?php esc_html_e( 'Authors', 'noptin-addons-pack' ); ?></strong>
					<input type="text" name="noptin_email[noptin-ap-authors]" class="widefat" placeholder="<?php esc_attr_e( 'Send for all authors', 'noptin-addons-pack' ); ?>" value="<?php echo esc_attr( $authors ); ?>">

					<span class="noptin-help-text">
						<?php esc_html_e( 'Enter a comma-separated list of author IDs or leave blank to send for all authors.', 'noptin-addons-pack' ); ?>
					</span>

				</label>

			</p>

			<?php if ( ! empty( $available_languages ) && is_array( $available_languages ) ) : ?>
				<p class="noptin-automation-post-type">

					<label style="display: block;">
						<strong><?php esc_html_e( 'Language', 'noptin-addons-pack' ); ?></strong>
						<select name="noptin_email[noptin-ap-language]" style="width: 100%;" class="noptin-ap-language-select">
							<option value="" <?php selected( empty( $language ) ); ?> disabled><?php esc_html_e( 'All languages', 'noptin-addons-pack' ); ?></option>
							<?php foreach ( $available_languages as $id => $name ) : ?>
								<option value="<?php echo esc_attr( $id ); ?>" <?php selected( $id, $language ); ?>><?php echo esc_html( $name ); ?></option>
							<?php endforeach; ?>
						</select>

						<span class="noptin-help-text">
							<?php esc_html_e( 'Optional. Send for posts in the selected language only.', 'noptin-addons-pack' ); ?>
						</span>

					</label>

				</p>
			<?php endif; ?>
		<?php endif; ?>

		<?php

    }

	/**
     * Returns the post type for a campaign
     *
     * @param   Noptin_Newsletter_Email $campaign
     * @since   1.0.0
     * @return  string
     */
    public function get_campaign_post_type( $campaign ) {

        $post_type = $campaign->get( 'noptin-ap-post-type' );
		return empty( $post_type ) ? 'post' : sanitize_text_field( $post_type );

	}

	/**
     * Filters campaign post types
     *
     * @access      public
     * @since       1.0.0
     * @return      string
     */
    public function filter_campaign_post_type( $post_type, $campaign ) {
		return $this->get_campaign_post_type( $campaign );
	}

	/**
	 * Filters valid for post
	 *
	 * @param bool $valid
	 * @param Noptin_Automated_Email $automation
	 * @param WP_Post $post
	 */
	public function filter_valid_for_post( $valid, $campaign, $post ) {

		// If the post type is not valid, return false.
		if ( $this->get_campaign_post_type( $campaign ) !== $post->post_type ) {
			return false;
		}

		// Language.
		$language  = $campaign->get( 'noptin-ap-language' );

		if ( ! empty( $language ) && function_exists( 'pll_get_post_language' ) && pll_get_post_language( $post->ID, 'locale' ) !== $language && pll_get_post_language( $post->ID ) !== $language ) {
			return false;
		}

		// If the author is not valid, return false.
		$authors = array_filter( noptin_parse_list( $campaign->get( 'noptin-ap-authors', array() ) ) );
		if ( ! empty( $authors ) ) {

			$author_ids = array();
			foreach ( $authors as $author ) {

				// Return false if author is excluded.
				if ( '-' === substr( $author, 0, 1 ) && trim( substr( $author, 1 ) ) === $post->post_author ) {
					return false;
				}

				if ( '-' !== substr( $author, 0, 1 ) ) {
					$author_ids[] = trim( $author );
				}
			}

			if ( ! empty( $author_ids ) && ! in_array( $post->post_author, $author_ids, true ) ) {
				return false;
			}
		}

		return true;

	}

	/**
	 * Filters the post digest query.
	 *
	 * @param  array $query
	 * @param array $args
	 * @param Noptin_Post_Digest $digest
	 * @return array
	 */
	public function filter_post_digest_query( $query, $args, $digest ) {

		// Default to posts.
		$post_type = 'post';

		// If we have a post digest, use its post type.
		if ( ! empty( $digest->post_digest ) ) {
			$post_type = $this->get_campaign_post_type( $digest->post_digest );
		}

		// If a post type was manually passed, use it.
		if ( ! empty( $args['post_type'] ) ) {
			$post_type = $args['post_type'];
		}

		$query['post_type'] = noptin_parse_list( $post_type );

		// Set other query args.
		$advanced_args = array(
			'author'         => 'int',
			'author_name'    => 'string',
			'author__in'     => 'array_int',
			'author__not_in' => 'array_int',
			'orderby'        => 'string',
			'order'          => 'string',
			'meta_key'       => 'string',
			'meta_value'     => 'string',
			'meta_value_num' => 'float',
			'meta_compare'   => 'string',
		);

		// Allow to filter post digests by language.
		if ( function_exists( 'pll_languages_list' ) ) {
			$advanced_args['lang'] = 'string';
		}

		foreach ( $advanced_args as $key => $type ) {

			if ( ! empty( $args[ $key ] ) ) {

				$value = $args[ $key ];

				if ( 'array_int' === $type ) {
					$value = noptin_parse_int_list( $value );
				} elseif ( 'int' === $type ) {
					$value = intval( $value );
				} elseif ( 'float' === $type ) {
					$value = floatval( $value );
				}

				$query[ $key ] = $value;

			}
		}

		return $query;
	}

}

new Noptin_Custom_Post_Types();
