<?php
/**
 * Syncs subscriber to another site.
 *
 */

defined( 'ABSPATH' ) || exit;

/**
 * Syncs the subscriber to another site.
 *
 * @since       1.0.0
 */
class Noptin_Sync_Action extends Noptin_Abstract_Action {

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'noptin_sync_subscriber_action';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'Sync Subscriber', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'Add or update the subscriber on another Noptin install', 'noptin-addons-pack' );
	}

	/**
	 * Retrieve the actions's rule table description.
	 *
	 * @since 1.11.9
	 * @param Noptin_Automation_Rule $rule
	 * @return array
	 */
	public function get_rule_table_description( $rule ) {

		$settings = $rule->action_settings;

		if ( empty( $settings['remote_url'] ) ) {
			return sprintf(
				'<span class="noptin-rule-error">%s</span>',
				esc_html__( 'Error: Remote URL not specified', 'noptin-addons-pack' )
			);
		}

		if ( empty( $settings['remote_secret'] ) ) {
			return sprintf(
				'<span class="noptin-rule-error">%s</span>',
				esc_html__( 'Error: Remote secret not specified', 'noptin-addons-pack' )
			);
		}

		// Obfuscate half of the secret.
		$secret = $settings['remote_secret'];
		$length = strlen( $secret );
		$secret = substr( $secret, 0, $length / 2 ) . str_repeat( '*', $length / 2 );

		return $this->rule_action_meta(
			array(
				__( 'Remote URL', 'noptin-addons-pack' )    => esc_url_raw( $settings['remote_url'] ),
				__( 'Remote Secret', 'noptin-addons-pack' ) => $secret,
			),
			$rule
		);

	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array(
			'remote_url'    => array(
				'el'          => 'input',
				'label'       => __( 'Remote Sync URL', 'noptin-addons-pack' ),
				'description' => __( "The remote site's sync URL", 'noptin-addons-pack' ),
			),
			'remote_secret' => array(
				'el'          => 'input',
				'type'        => 'text',
				'label'       => __( 'Remote Sync Secret', 'noptin-addons-pack' ),
				'description' => __( "The remote site's sync secret", 'noptin-addons-pack' ),
			),
		);

	}

	/**
	 * Enrolls the user in a course.
	 *
	 * @since 1.0.0
	 * @param Noptin_Subscriber $subscriber The subscriber.
	 * @param Noptin_Automation_Rule $rule The automation rule used to trigger the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function run( $subscriber, $rule, $args ) {

		if ( empty( $rule->action_settings['remote_url'] ) ) {
			return;
		}

		$subscriber_data         = is_callable( array( $subscriber, 'to_array' ) ) ? $subscriber->to_array() : $subscriber->get_data();
		$subscriber_data['meta'] = $subscriber->get_meta();
		$subscriber_data['args'] = array();

		if ( false !== wp_json_encode( $args ) ) {
			$subscriber_data['args'] = $args;
		}

		$subscriber_data['secret'] = $rule->action_settings['remote_secret'];

		if ( ! empty( $subscriber_data['meta']['_subscriber_via'] ) && is_numeric( $subscriber_data['meta']['_subscriber_via'] ) ) {
			$subscriber_data['meta']['_subscriber_via'] = get_the_title( $subscriber_data['meta']['_subscriber_via'] );
		}

		if ( isset( $subscriber_data['second_name'] ) ) {
			$subscriber_data['last_name'] = $subscriber_data['second_name'];
			unset( $subscriber_data['second_name'] );
		}

		wp_remote_post(
			$rule->action_settings['remote_url'],
			array(
				'body'        => wp_json_encode( $subscriber_data ),
				'headers'     => array( 'Content-Type' => 'application/json' ),
				'data_format' => 'body',
			)
		);

	}

}
