<?php
/**
 * Noptin.com Custom Post Type Notifications Class.
 *
 * @package Noptin\noptin.com
 * @since   1.5.1
 */

defined( 'ABSPATH' ) || exit;

/**
 * Noptin_Custom_Taxonomies Class
 *
 * @since 1.5.1
 * @ignore
 */
class Noptin_Custom_Taxonomies {

	/**
	 * Init variables.
	 *
	 * @since       1.0.0
	 */
	public function __construct() {

		// Display metabox.
		add_action( 'noptin_automated_email_post_notifications_options', array( $this, 'render_metabox' ) );
		add_action( 'noptin_automated_email_post_digest_options', array( $this, 'render_metabox' ) );

		add_filter( 'noptin_new_post_notification_valid_for_post', array( $this, 'filter_campaign_terms' ), 10, 3 );
		add_filter( 'noptin_post_digest_posts_query', array( $this, 'filter_post_digest_query' ), 10, 3 );

		// Misc.
		add_filter( 'noptin_post_digest_merge_tag_query', array( $this, 'filter_post_digest_merge_tag_query' ), 20, 3 );
    }

	/**
     * Post terms select box.
     *
     * @access      public
     * @since       1.0.0
     * @return      string
     */
    public function render_metabox( $campaign ) {

		$terms 		  = $this->get_campaign_terms( $campaign );
		$terms_action = $this->get_campaign_terms_action( $campaign );

		?>

		<div class="noptin-automation-post-terms">

			<p>
				<label style="display: block;">
					<strong><?php esc_html_e( 'Filter by terms', 'noptin-addons-pack' ); ?></strong>
					<select name="noptin_email[noptin-ap-terms-action]" style="width: 100%;" class="noptin-ap-post-terms-action-select">
						<?php foreach ( $this->get_campaign_term_actions() as $key => $action ) : ?>
							<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $key, $terms_action ); ?>><?php echo esc_html( $action ); ?></option>
						<?php endforeach; ?>
					</select>
					<span class="noptin-help-text noptin-ap-post-terms-help-no-filter">
						<?php if ( 'post_digest' !== $campaign->type ) : ?>
							<?php esc_html_e( 'Optional. Select the tags, categories, and terms to send new post notifications for.', 'noptin-addons-pack' ); ?>
						<?php endif; ?>
					</span>
				</label>
			</p>

			<div class="noptin-ap-post-terms-checkbox-wrapper">

				<ul class="noptin-checkbox-zone">

					<?php foreach ( array_keys( noptin_get_post_types() ) as $post_type ) : ?>
						<?php foreach ( $this->get_post_type_terms( $post_type ) as $term ) : ?>
							<li data-type="<?php echo esc_attr( $post_type ); ?>">
								<label>
									<input
										name='noptin_email[noptin-ap-terms][<?php echo esc_attr( $term->term_taxonomy_id ); ?>][]'
										type='checkbox'
										value='<?php echo esc_attr( $term->term_id ); ?>'
										<?php checked( in_array( (int) $term->term_id, $terms, true ) ); ?>
									>
									<span><?php echo esc_html( "{$term->name} &mdash; {$term->taxonomy}" ); ?></span>
								</label>
							</li>
						<?php endforeach; ?>
					<?php endforeach; ?>

				</ul>

				<?php if ( 'post_digest' === $campaign->type ) : ?>
					<span class="noptin-help-text">

						<span class="noptin-ap-post-terms-help-include">
							<?php esc_html_e( 'The [[post_digest]] merge tag will ONLY display posts with the selected terms unless you set a post type and taxonomy in the merge tag.', 'noptin-addons-pack' ); ?>
						</span>

						<span class="noptin-ap-post-terms-help-exclude">
							<?php esc_html_e( 'The [[post_digest]] merge tag will NOT display posts with the selected terms unless you set a post type and taxonomy in the merge tag.', 'noptin-addons-pack' ); ?>
						</span>

						<a href="https://noptin.com/guide/sending-emails/new-post-notifications/#limit-by-taxonomies" target="_blank"><?php esc_html_e( 'Learn more', 'noptin-addons-pack' ); ?></a>
					</span>
				<?php endif; ?>
			</div>

		</div>

		<script>

			// Hide and uncheck irrelevant terms when the post type changes.
			var post_type_changed = function() {

				var wrapper   = jQuery( '.noptin-ap-post-terms-checkbox-wrapper' );
				var post_type = jQuery( '.noptin-ap-post-type-select' ).val();

				// Hide irrelevant terms.
				wrapper
					.find( 'li' )
					.each( function() {
						var list = jQuery( this );

						if ( post_type != list.data( 'type' ) ) {
							list.hide();
							list.find( 'input' ).prop( 'checked', false );
						} else {
							list.show();
						}

					});

			}

			// Fired when the term action changes.
			var term_action_changed = function() {

				var action = jQuery('.noptin-ap-post-terms-action-select').val();

				if ( action == '-1' ) {
                	jQuery( '.noptin-ap-post-terms-checkbox-wrapper' ).hide();
					jQuery( '.noptin-ap-post-terms-help-no-filter' ).show();
					jQuery( '.noptin-ap-post-terms-help-exclude, .noptin-ap-post-terms-help-include' ).hide();
            	} else if ( action == 'include' ) {
					jQuery( '.noptin-ap-post-terms-help-include' ).show();
					jQuery( '.noptin-ap-post-terms-help-exclude, .noptin-ap-post-terms-help-no-filter' ).hide();
                	jQuery( '.noptin-ap-post-terms-checkbox-wrapper' ).show();
            	} else {
					jQuery( '.noptin-ap-post-terms-help-include, .noptin-ap-post-terms-help-no-filter' ).hide();
					jQuery( '.noptin-ap-post-terms-help-exclude' ).show();
                	jQuery( '.noptin-ap-post-terms-checkbox-wrapper' ).show();
            	}

			}

			jQuery(document).ready(function() {

				post_type_changed();
				term_action_changed();
				jQuery( '.noptin-ap-post-type-select' ).on( 'change', post_type_changed );
				jQuery( '.noptin-ap-post-terms-action-select' ).on( 'change', term_action_changed );

			})

		</script>
		<?php

    }

	/**
     * Returns the terms for a campaign
     *
     * @param  Noptin_Newsletter_Email $campaign
	 * @param  bool $grouped
     * @since  1.0.0
     * @return array
     */
    public function get_campaign_terms( $campaign, $grouped = false ) {
		$terms = $campaign->get( 'noptin-ap-terms' );

		if ( empty( $terms ) || ! is_array( $terms ) ) {
			return array();
		}

		if ( $grouped ) {
			return array_filter( $terms );
		}

		$flat = array();

		foreach ( $terms as $_terms ) {
			$flat = array_merge( $flat, wp_parse_id_list( $_terms ) );
		}

		return array_filter( $flat );

	}

	/**
     * Returns the terms action.
     *
     * @param  Noptin_Newsletter_Email $campaign
     * @since  1.0.0
     * @return string
     */
    public function get_campaign_terms_action( $campaign ) {
		$actions	  = $this->get_campaign_term_actions();
		$terms_action = $campaign->get( 'noptin-ap-terms-action' );
		return isset( $actions[ $terms_action ] ) ? $terms_action : '-1';
	}

	/**
     * Returns the term actions
     *
     * @access      public
     * @since       1.0.0
     * @return      array
     */
    public function get_campaign_term_actions() {
		return array(
			'-1'      => __( 'Do not filter by tags, categories, etc', 'noptin-addons-pack' ),
			'include' => __( 'Only send for some tags, categories, etc', 'noptin-addons-pack' ),
			'exclude' => __( 'Do not send for some tags, categories, etc', 'noptin-addons-pack' ),
		);
	}

	/**
     * Returns all the terms for a given post type
     *
     * @access      public
     * @since       1.0.0
     * @return      WP_Term[]
     */
    public function get_post_type_terms( $post_type ) {

		$taxonomies = get_object_taxonomies( $post_type );

		if ( empty( $taxonomies ) ) {
			return array();
		}

		return $this->get_all_terms( $taxonomies );

	}

	/**
     * Returns all the terms
     *
     * @access      public
     * @since       1.0.0
     * @return      WP_Term[]
     */
    protected function get_all_terms( $taxonomies ) {

		$terms = get_terms(
			array(
				'taxonomy'   => $taxonomies,
				'hide_empty' => false,
			)
		);

		return is_array( $terms ) ? $terms : array();

    }

	/**
     * Filters campaign terms
     *
     * @param bool                   $valid
	 * @param Noptin_Automated_Email $automation
	 * @param WP_Post                $post
     * @since                        1.0.0
     * @return                       bool
     */
    public function filter_campaign_terms( $valid, $automation, $post ) {

		if ( ! $valid ) {
			return $valid;
		}

		$terms 		  = $this->get_campaign_terms( $automation );
		$terms_action = $this->get_campaign_terms_action( $automation );

		if ( '-1' === $terms_action || empty( $terms ) ) {
			return $valid;
		}

		$taxonomies = get_object_taxonomies( get_post_type( $post ) );
		$post_terms = wp_get_object_terms( $post->ID, $taxonomies );
		$found		= false;

		if ( is_wp_error( $post_terms ) ) {
            return $valid;
        }

		foreach ( wp_list_pluck( $post_terms, 'term_id', 'term_id' ) as $id ) {

			if ( in_array( (int) $id, $terms, true ) ) {
				$found = true;
				break;
			}
		}

		if ( 'exclude' === $terms_action && $found ) {
			return false;
		}

		if ( 'include' === $terms_action && ! $found ) {
			return false;
		}

		return $valid;

	}

	/**
     * Filters post digests query.
     *
     * @param array                  $query
	 * @param Noptin_Automated_Email $automation
     * @since                        1.0.0
     * @return                       bool
     */
    public function filter_post_digest_query( $query, $automation ) {
		global $wpdb;

		$terms 		  = $this->get_campaign_terms( $automation, true );
		$terms_action = $this->get_campaign_terms_action( $automation );

		if ( '-1' === $terms_action || empty( $terms ) ) {
			return $query;
		}

		$term_taxonomy_ids = implode( ', ', wp_parse_id_list( array_keys( $terms ) ) );
		$query['include']  = $wpdb->get_col( "SELECT DISTINCT `object_id` FROM `{$wpdb->term_relationships}` WHERE `term_taxonomy_id` IN ($term_taxonomy_ids)" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return $query;
	}

	/**
	 * Filters the post digest query.
	 *
	 * @param  array $query
	 * @param array $args
	 * @param Noptin_Post_Digest $digest
	 * @return array
	 */
	public function filter_post_digest_merge_tag_query( $query, $args, $digest ) {

		// Filter terms.
		if ( empty( $args['post_type'] ) ) {
			return $this->filter_post_digest_query( $query, $digest->post_digest );
		}

		// Set the taxonomy query.
		$tax_query  = array();
		$post_types = noptin_parse_list( $query['post_type'] );

		foreach ( noptin_parse_list( $query['post_type'] ) as $post_type ) {

			foreach ( get_object_taxonomies( $post_type ) as $taxonomy ) {

				// Special treatment for tags.
				if ( 'post_tag' === $taxonomy ) {

					$allowed = array(
						'tag'           => 'string',
						'tag_id'        => 'int',
						'tag__and'      => 'array_int',
						'tag__in'       => 'array_int',
						'tag__not_in'   => 'array_int',
						'tag_slug__and' => 'array_string',
						'tag_slug__in'  => 'array_string',
					);

					foreach ( $allowed as $key => $type ) {
						if ( ! empty( $args[ $key ] ) ) {
							$value = $args[ $key ];
						} elseif ( ! empty( $args[ 'post_' . $key ] ) ) {
							$value = $args[ 'post_' . $key ];
						} else {
							continue;
						}

						if ( 'array_int' === $type ) {
							$value = wp_parse_id_list( $value );
						} elseif ( 'array_string' === $type ) {
							$value = noptin_parse_list( $value, true );
						} elseif ( 'int' === $type ) {
							$value = absint( $value );
						}

						$query[ $key ] = $value;
					}
					continue;
				}

				// Special treatment for categories.
				if ( 'category' === $taxonomy ) {

					$allowed = array(
						'cat'              => 'string',
						'category_name'    => 'string',
						'category__and'    => 'array_int',
						'category__in'     => 'array_int',
						'category__not_in' => 'array_int',
					);

					if ( isset( $args['category'] ) ) {
						$args['category_name'] = $args['category'];
					}

					foreach ( $allowed as $key => $type ) {
						if ( ! empty( $args[ $key ] ) ) {
							$value = $args[ $key ];
						} elseif ( ! empty( $args[ 'post_' . $key ] ) ) {
							$value = $args[ 'post_' . $key ];
						} else {
							continue;
						}

						if ( 'array_int' === $type ) {
							$value = wp_parse_id_list( $value );
						}

						$query[ $key ] = $value;
					}
					continue;
				}

				// Taxonomy slugs.
				if ( isset( $args[ $taxonomy ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => noptin_parse_list( $args[ $taxonomy ], true ),
					);
				}

				// Taxonomy ids.
				if ( isset( $args[ $taxonomy . '_ids' ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'term_id',
						'terms'    => noptin_parse_int_list( $args[ $taxonomy . '_ids' ] ),
					);
				}

				// Taxonomy slugs not in.
				if ( isset( $args[ $taxonomy . '_not_in' ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => noptin_parse_list( $args[ $taxonomy . '_not_in' ], true ),
						'operator' => 'NOT IN',
					);
				}

				// Taxonomy ids not in.
				if ( isset( $args[ $taxonomy . '_ids_not_in' ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'term_id',
						'terms'    => noptin_parse_int_list( $args[ $taxonomy . '_ids_not_in' ] ),
						'operator' => 'NOT IN',
					);
				}

				// Taxonomy slugs AND.
				if ( isset( $args[ $taxonomy . '_and' ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => noptin_parse_list( $args[ $taxonomy . '_and' ], true ),
						'operator' => 'AND',
					);
				}

				// Taxonomy ids AND.
				if ( isset( $args[ $taxonomy . '_ids_and' ] ) ) {
					$tax_query[] = array(
						'taxonomy' => $taxonomy,
						'field'    => 'term_id',
						'terms'    => noptin_parse_int_list( $args[ $taxonomy . '_ids_and' ] ),
						'operator' => 'AND',
					);
				}
			}
		}

		if ( ! empty( $tax_query ) ) {

			if ( 1 < count( $tax_query ) ) {
				$tax_query['relation'] = count( $post_types ) > 1 ? 'OR' : 'AND';
			}

			$query['tax_query'] = $tax_query;
		}

		return $query;

	}
}

new Noptin_Custom_Taxonomies();
